#include <stdio.h>
#include <fcgiapp.h>
#include <sys/time.h>
#include <errno.h>
#include <string.h>
#include <ctype.h>
#include <math.h>
#include "lstring.h"
#include "sql.h"
#include "vm.h"
#include "utils.h"
#include "params.h"
#include "main.h"
#include "blang.h"


static int evaluate(INSTR const, const struct timeval* const);
static int copyval(const VAL restrict, const VAL restrict);
static void copyhash(ht **dst, ht * const src);
static int assignarr(VAL, size_t);
static int getarr(VAL, size_t);
static int executequery(size_t);
static int tostderr(size_t);
static char *getstring(const VAL);
static char* tostring(const VAL);
static bool tobool(const VAL);
static long double toldbl(const VAL);
static void sendjson(size_t);
static void buildjson(STRING* const, size_t);
static void buildarray(STRING* const);
static int getpost(const char[]);
static void push(const VAL);
static void pop(void);
static void freestack(void);
static int compare(const VAL restrict, const VAL restrict);
static int substring(void);
static int concatenate(const VAL restrict, const VAL restrict);
static int evaluatearithmetic(enum opcode, const VAL restrict, const VAL restrict);
static int evaluateneg(const VAL restrict);
static int evaluatelogical(enum opcode, const VAL restrict, const VAL restrict);
static void evaluatenot(const VAL restrict);
static int changecase(const VAL, const int);
static void getmillis(const struct timeval* const);
#ifdef DEBUG
struct timeval* timerstart(void);
static void timer(struct timeval * const);
static void displayinstr(INSTR);
static char* maddr(VAL);
#endif

char ERRORJSON[] = STR_ERRORJSON;


struct col{
	char *key;
	char *index;
};

// the stack
static struct{
	VAL stack;
	size_t size;
} stack;
static VAL stackptr;



int srcline; // line of script source being processed
int ierror;  // if 1 program will halt
extern struct stvalue ax, bx;  // the two registers of the VM

extern struct stblock program;  // linked list of instructions; the program itself
extern struct pargs *post;  // http post values will be stored here
extern FCGX_Request request;
extern struct stconfig config;

/******************************************************************************
 *
 * initialize stack and registers
 *
 *****************************************************************************/

void initvm(void){
	stack.size = STACK_INITIAL_SIZE;
	stack.stack = malloc(stack.size * sizeof(struct stvalue));
	stackptr = stack.stack - 1;

	ax.type = bx.type = VUNDEF;
}

/******************************************************************************
 *
 * executes the program
 * directly manages every jump or flow control instructions
 * parameters: tm: start of program
 * returns 1 on program failure, 0 on success
 *
 *****************************************************************************/
/*









	TODO: RETURN ERROR TO BROWSER







*/
int runVM(const struct timeval * const tm){
#ifdef DEBUG
	for(INSTR instr = program.first; instr; instr = instr->next)
		displayinstr(instr);
#endif
	srcline = 0;
	ierror = 0;
	INSTR instr = program.first;
	while(instr){
		if(ierror){
			freestack();
			return 1;
		}
		srcline = instr->srcline;
		switch(instr->op){
		case IEND:
			return 0;
		case IJMP:
			instr = instr->jump.jmpto;
			continue;
		case IJZ:
			if(tobool(instr->jump.testval) == false) {
				instr = instr->jump.jmpto;
			}
			else instr = instr->next;
			continue;
		case IRETURN:
			sendjson(instr->pop);
			return 0;
		default:
			break;
		}
		if(evaluate(instr, tm)) return 1;
		instr = instr->next;
	}
	return 0;
}

/******************************************************************************
 *
 * decodes executes an instruction
 * params: instr: the instruction
 *         tm:    program start time
 * returns: 1 on error, 0 on success
 *
 *****************************************************************************/

static int evaluate(INSTR const instr, const struct timeval * const tm){

	switch(instr->op){
		case IPUSH:
			push(instr->pushval);
			break;
		case IEXECQ:
			return executequery(instr->pop);
		case IDEBUG:
			return tostderr(instr->pop);
		case IMILLIS:
			getmillis(tm);
			break;
		case IUPPER:
			return changecase(instr->opds.op1, 1);
		case ILOWER:
			return changecase(instr->opds.op1, 0);
		case ISUBSTR:
			return substring();
		case IPOST:
			return getpost(instr->postname);
		case IASSIGN:
			freevarval(instr->opds.op1);
			return copyval(instr->opds.op1, instr->opds.op2);
		case IASSIGNARR:
			return assignarr(instr->array.array, instr->array.pop);
		case IEQUALS:
			{ // allows decl following switch label
				const int res = compare(instr->opds.op1, instr->opds.op2);
				if (res == -2) return 1;
				freevarval(&ax);
				ax.boolval = res ? false : true;
				ax.type = VBOOL;
			}
			break;
		case INEQUALS:
			{ // allows decl following switch label
				const int res = compare(instr->opds.op1, instr->opds.op2);
				if (res == -2) return 1;
				freevarval(&ax);
				ax.boolval = res ? true : false;
				ax.type = VBOOL;
			}
			break;
		case ISUPEQ:
			{ // allows decl following switch label
				const int res = compare(instr->opds.op1, instr->opds.op2);
				if (res == -2) return 1;
				freevarval(&ax);
				ax.boolval = res == -1 ? false : true;
				ax.type = VBOOL;
			}
			break;
		case ISUP:
			{ // allows decl following switch label
				const int res = compare(instr->opds.op1, instr->opds.op2);
				if (res == -2) return 1;
				freevarval(&ax);
				ax.boolval = res == 1 ? true : false;
				ax.type = VBOOL;
			}
			break;
		case IINFEQ:
			{ // allows decl following switch label
				const int res = compare(instr->opds.op1, instr->opds.op2);
				if (res == -2) return 1;
				freevarval(&ax);
				ax.boolval = res == 1 ? false : true;
				ax.type = VBOOL;
			}
			break;
		case IINF:
			{ // allows decl following switch label
				const int res = compare(instr->opds.op1, instr->opds.op2);
				if (res == -2) return 1;
				freevarval(&ax);
				ax.boolval = res == -1 ? true : false;
				ax.type = VBOOL;
			}
			break;
		case IADD:
		case ISUB:
		case IMUL:
		case IDIV:
			return evaluatearithmetic(instr->op, instr->opds.op1, instr->opds.op2);
		case ILOR:
		case ILAND:
			return evaluatelogical(instr->op, instr->opds.op1, instr->opds.op2);
		case INEG:
			return evaluateneg(instr->opds.op1);
		case INOT:
			evaluatenot(instr->opds.op1);
			return 0;
		case IARR:
			return getarr(instr->array.array, instr->array.pop);
		default:
			errprint("UNKNOWN OPCODE %d", instr->op);
			return 1;
	}
	return 0;
}


/******************************************************************************
 * change case of input and put it to ax
 * params: VAL:   the value to change
 *         icase: 0 for lowercase, 1 for uppercase
 * returns: 1 on error, 0 on success
 *
 *****************************************************************************/

static int changecase(const VAL val, const int icase){
	freevarval(&ax);
	char *c = tostring(val);
	if(!c){
		errprint("not an string");
		ierror = 1;
		return 1;
	}
	ax.type = VSTRING;
	ax.strval = c;
	while(*c != '\0'){
		*c = icase == 0 ? tolower(*c) : toupper(*c);
		c++;
	}
	return 0;
}

/******************************************************************************
 *
 * put milliseconds since script start into ax
 *
 *****************************************************************************/


static void getmillis(const struct timeval * const tm){
	struct timeval now, res;
	gettimeofday(&now, NULL);
	timersub(&now, tm, &res);
	freevarval(&ax);
	ax.type = VLDOUBLE;
	ax.ldblval = res.tv_sec * 1000 + res.tv_usec / 1000;
}

/******************************************************************************
 *
 * reset stack
 *
 *****************************************************************************/

static void freestack(void){
	while(stackptr >= stack.stack){
		freevarval(stackptr);
		stackptr--;
	}
}


/******************************************************************************
 *
 * push a value on the stack
 *
 *****************************************************************************/

static void push(const VAL val){
	stackptr++;
	if(stackptr == stack.stack + stack.size){
		stack.stack = reallocarray(stack.stack, stack.size + STACK_GROW_SIZE, sizeof(struct stvalue));
		stackptr = stack.stack + stack.size;
		stack.size = stack.size + STACK_GROW_SIZE;
	}
	copyval(stackptr, val);
}



/******************************************************************************
 *
 * frees a value from the stack (and decrease stackptr)
 *
 *****************************************************************************/

static void pop(void){
	// allocated stack space never shrinks, due to the nature of the program,
	// reaching highest value again is highly probable, so it would incur a performance penalty
	freevarval(stackptr);
	stackptr--;
}



/******************************************************************************
 *
 * manage logical operand, put the result to ax
 * returns: 1 on error, 0 on success
 *
 *****************************************************************************/

static int evaluatelogical(enum opcode op, const VAL restrict v1, const VAL restrict v2){
	const bool b1 = tobool(v1);
	const bool b2 = tobool(v2);

	freevarval(&ax);
	ax.type = VBOOL;
	switch(op){
	case ILOR:
		ax.boolval = b1 || b2;
		break;
	case ILAND:
		ax.boolval = b1 && b2;
		break;
	default:
		errprint("this is a bug");
		ierror = 1;
		return 1;
	}
	return 0;
}

/******************************************************************************
 *
 * manage not operand, result to ax
 *
 *****************************************************************************/


static void evaluatenot(const VAL restrict v1){
	freevarval(&ax);
	ax.type = VBOOL;
	ax.boolval = !tobool(v1);
}


/******************************************************************************
 *
 * manage negative op, result to ax
 * returns: 1 on error, 0 on success
 *
 *****************************************************************************/

static int evaluateneg(const VAL restrict v1){
	long double op1 = toldbl(v1);
	if(isnan(op1)){
		errprint("Not a number");
		ierror = 1;
		return 1;
	}
	freevarval(&ax);
	ax.type = VLDOUBLE;
	ax.ldblval = -op1;
	return 0;
}




/******************************************************************************
 *
 * manage basic arithmetic operations, result to ax
 * an exception is made on "+", we try to perform a concatenation
 * if concersion to number of any of the provided values fail
 * returns: 1 on error, 0 on success
 *
 *****************************************************************************/

static int evaluatearithmetic(enum opcode op, const VAL restrict v1, const VAL restrict v2){
	long double op1 = toldbl(v1);
	long double op2 = toldbl(v2);

	if(isnan(op1) || isnan(op2)){
		if(op == IADD) return concatenate(v1, v2);
		ierror = 1;
		return 1;
	}
	freevarval(&ax);
	ax.type = VLDOUBLE;
	switch(op){
	case ISUB:
		ax.ldblval = op1 - op2;
		break;
	case IMUL:
		ax.ldblval = op1 * op2;
		break;
	case IDIV:
		ax.ldblval = op1 / op2;
		break;
	case IADD:
		ax.ldblval = op1 + op2;
		break;
	default:
		errprint("this is a bug");
		ierror = 1;
		return 1;
	}

	return 0;
}


/******************************************************************************
 *
 * substr(string, start, end) (params are provided from the stack)
 * computed substring is stored in ax
 * returns: 1 on error, 0 on success
 *
 *****************************************************************************/


static int substring(void){
	freevarval(&ax);
	long double fstart, fend;
	char *c = tostring(stackptr);
	int i, start, end;
	pop();
	fstart = toldbl(stackptr);
	pop();
	fend = toldbl(stackptr);
	pop();
	if(!c || isnan(fstart) || isnan(fend) || fstart < 0 || fend < 0){
		errprint("invalid parameters");
		ierror = 1;
		free(c);
		return 1;
	}
	ax.type = VSTRING;
	start = fstart;
	end = fend;
	if(strlen(c) < start){
		ax.strval = strdup("");
		free(c);
		return 0;
	}
	ax.strval = malloc(strlen(c + start) + 1);
	for(i = 0; c[i + start] != '\0' && i < end; i++)
		ax.strval[i] = c[i + start];

	ax.strval[i] = '\0';
	free(c);
	return 0;
}

/******************************************************************************
 *
 * gets post string from key (into ax)
 * returns 0 success, 1 if key not found
 *****************************************************************************/

static int getpost(const char key[]){
	struct pargs *currentpost = post;
	while(currentpost){
		if(!strcmp(currentpost->key, key)) break;
		currentpost = currentpost->next;
	}
	if(!currentpost) {
		errprint("POST value \"%s\" not found", key);
		ierror = 1;
		return 1;
	}
	freevarval(&ax);
	ax.type = VSTRING;
	ax.strval = strdup(currentpost->val);
	return 0;
}


/******************************************************************************
 *
 * manage concatenation of two values
 * result put to ax
 * returns: 1 on error, 0 on success
 *
 *****************************************************************************/


static int concatenate(const VAL restrict v1, const VAL restrict v2){
	char *c1 = tostring(v1);
	if(!c1) {
		errprint("value is not a string");
		ierror = 1;
		return 1;
	}
	char *c2 = tostring(v2);
	if(!c2) {
		free(c1);
		errprint("value is not a string");
		ierror = 1;
		return 1;
	}
	const size_t l1 = strlen(c1);
	const size_t l2 = strlen(c2) + 1;
	// !!! v1 or v2 can be ax
	char *c = malloc(l1 + l2);
	memcpy(c, c1, l1);
	memcpy(c + l1, c2, l2);
	free(c1);
	free(c2);
	freevarval(&ax);
	ax.type = VSTRING;
	ax.strval = c;
	return 0;
}



/******************************************************************************
 *
 * compares two operands
 * returns: -1 if v1 < v2
 *           0 if v1 = v2
 *           1 if v1 > v2
 *          -2 on error
 *
 *****************************************************************************/

static int compare(const VAL restrict v1, const VAL restrict v2){
	int ret;

	switch(v1->type){
	case VUNDEF:
		if(v2->type == VUNDEF) ret = 0;
		else ret = -1;
	break;
	case VBOOL:
	case VSTRING:
	case VLDOUBLE:
		{
			if(v2->type == VUNDEF){
				ret = 1;
				break;
				goto compareerror; // aaaah the goto dogma.
				                   // this is actually a perfectly fine case of use
			}
			long double a, b;
			a = toldbl(v1);
			b = toldbl(v2);
			if(!isnan(a) && !isnan(b)){
				if(a < b) ret = -1;
				else if(a > b) ret = 1;
				else ret = 0;
				break;
			}
			char *s1 = tostring(v1);
			if(!s1) goto compareerror;
			char *s2 = tostring(v2);
			if(!s2){
				free(s1);
				goto compareerror;
			}
			ret = strcmp(s1, s2);
			free(s1);
			free(s2);
		}
		break;
	case VARRAY:
		if(v2->type == VUNDEF){
			ret = -1;
			break;
		}
		goto compareerror; // aaaah the goto dogma.
		                   // this is actually a perfectly fine case of use
	default:
		goto compareerror;

	}

	return ret;
compareerror:
	errprint("unmanaged type for comparison");
	ierror = 1;
	return -2;
}


/******************************************************************************
 *
 * copy value from src to dst (DOES NOT FREE dst)
 * !!! if copying from ax, ax will be set to VUNDEF !!!
 * !!! this is an optimization (ax is normally never reused) !!!
 * returns 1 on error, 0 on success
 *
 *****************************************************************************/

static int copyval(const VAL restrict dst, const VAL restrict src){
	if(dst == src) return 0; // or bad things will happen
	// direct struct copy cannot be slower than memcpy()
	// *dst = *src;
	// memcpy(dst, src, sizeof(struct stvalue)); ...but crashes when embedded in a flexible array member :-)
	*dst = *src; // ok removed flexible array

	// ax shall never be reused after its copy (remember this VM is not designed for handwritten pseudo asm);
	// hence we will directly transfer pointer contents,
	// thus avoiding costly copy and later free
	if(src == &ax){
		src->type = VUNDEF; // so future freevarval(ax) won't erase src's now pointed allocations
		return 0;
	}

	switch(src->type){
	case VSTRING:
		dst->strval = strdup(src->strval);
		break;
	case VARRAY:
		copyhash(&(dst->hash), src->hash);
		break;
	case VLDOUBLE:
	case VBOOL:
	case VUNDEF:
	case VJSONARRAY:  // for push, we will consider them only as ref
	case VJSONOBJECT:
		break;
	}

	return 0;
}

/******************************************************************************
 *
 * copies an hash from array value from src to dst
 * recursive
 *
 *****************************************************************************/


static void copyhash(ht **dst, ht * const src){
	if(src == NULL){
		*dst = src;
		return;
	}
	*dst = malloc(sizeof(struct hashtable));
	memcpy(*dst, src, sizeof(struct hashtable));
	(*dst)->entries = malloc(sizeof(struct hashtable_entry) * src->capacity);
	(*dst)->lastentry = (*dst)->entries + (*dst)->capacity - 1;

	hti *itersrc = hti_create(src);
	for(size_t i = 0; i < (*dst)->capacity; i++){
		if(itersrc->key && i == itersrc->current - src->entries){
			//offsetof will skip the value field
			memcpy((*dst)->entries + i, itersrc->current, offsetof(struct hashtable_entry, value));
			copyval(&((*dst)->entries[i].value), &(itersrc->current->value));
			hti_next(itersrc);
		}
		else (*dst)->entries[i].filled = 0;
	}
	free(itersrc);
}



/******************************************************************************
 *
 * assign a value to an array member (frees dst)
 * stack in order of pop:
 * dst value, src value, then indexes of array
 * parameters: nbparams: number of elements in stack
 * returns 1 on error, 0 on success
 *****************************************************************************/


static int assignarr(VAL dst, size_t nparams){
	struct stvalue emptynode;
	emptynode.type = VUNDEF;

	while(nparams > 1){

		if(dst->type != VARRAY) {
			freevarval(dst);
			dst->type = VARRAY;
			dst->hash = ht_create(DEFAULT_HASH_SIZE);
		}
		else if(dst->hash == NULL) dst->hash = ht_create(DEFAULT_HASH_SIZE);
		if(stackptr->type == VLDOUBLE) dst = ht_get(dst->hash, &stackptr->ldblval, 10, &emptynode);
		else {
			const size_t len = strlen(stackptr->strval);
			if(len > MAX_KEY_LENGTH){
				errprint("array index too long");
				do pop(); while(nparams--);
				return 1;
			}
			dst = ht_get(dst->hash, stackptr->strval, len, &emptynode);
		}
		pop();
		nparams--;
	}
	freevarval(dst);
	VAL src = stackptr;
	copyval(dst, src);
	pop();
	return 0;
}

/******************************************************************************
 *
 * gets a value from an array member (put to ax)
 * stack in order of pop:
 * src value, then indexes of array
 * parameters: nbparams: number of elements in stack
 * returns 1 on error, 0 on success
 *****************************************************************************/


static int getarr(VAL src, size_t nparams){
	freevarval(&ax);
	while(nparams--){
		if(src->type != VARRAY) {
			errprint("not an array");
			ierror = 1;
			return 1;
		}
		if(src->hash == NULL) return 0;
		if(stackptr->type == VLDOUBLE) src = ht_get(src->hash, &stackptr->ldblval, 10, NULL);
		else {
			const size_t len = strlen(stackptr->strval);
			if(len > MAX_KEY_LENGTH){
				errprint("array index too long");
				do pop(); while(nparams--);
				return 1;
			}
			src = ht_get(src->hash, stackptr->strval, strlen(stackptr->strval), NULL);
		}
		if(src == NULL){
			do pop(); while(nparams--);
			return 0;
		}
		pop();
	}
	copyval(&ax, src);
	return 0;
}

/******************************************************************************
 *
 * display nbargs stack elements to stderr (popped)
 * returns 1 on error, 0 on success
 *****************************************************************************/

static int tostderr(size_t nbargs){
	char *c;

	while(nbargs){
		c = tostring(stackptr);
		pop();
		nbargs--;
		if(config.socketpath) FCGX_PutS(c ? c : "N/A", request.err);
		else puts(c ? c : "N/A");
		free(c);
	}
	terminatestderr();
	return 0;
}


/******************************************************************************
 *
 * execute query (parameters are in the stack (popped))
 * a VARRAY result is the put into ax
 * parameters: nbparams: number of parematers to pop from the stack
 * returns: 1 on error, 0 on success
 *****************************************************************************/

int executequery(size_t nbparams){
	char **params;
	char *query = getstring(stackptr);
	if(!query){
		errprint("given query is not a string");
		for(size_t i = 0; i < nbparams; i++) pop();
		freevarval(&ax);
		ax.type = VUNDEF;
		return 1;
	}
	query = strdup(query);
	pop();
	nbparams--;

	if(nbparams){
		params = malloc(nbparams * sizeof(char*));

		for(size_t i = 0; i < nbparams; i++){
			params[i] = tostring(stackptr);
			if(params[i] == NULL) {
				errprint("query parameter %d is not a valid value", i + 1);
				for(size_t j = 0; j < i; j++) free(params[j]);
				free(params);
				free(query);
				while(i++ < nbparams) pop();
				freevarval(&ax);
				ax.type = VUNDEF;
				return 1;
			}
			pop();
		}
	}
	else params = NULL;
	int error;
	ht* results = vquery(query, params, nbparams, &error);
	free(query);

	for(size_t i = 0; i < nbparams; i++) free(params[i]);
	free(params);
	if(error == 1) {
		ierror = 1;
		return 1;
	}
	freevarval(&ax);
	ax.type = VARRAY;
	ax.hash = results;

	return 0;
}



/******************************************************************************
 *
 * get string representation of value (allocated)
 * returns: a string representation of v, NULL on failure
 ****************************************************************************/


static char* tostring(const VAL v){
	switch(v->type){
	case VSTRING:
		return strdup(v->strval);
	case VLDOUBLE:
		{
		int len = snprintf(NULL, 0, "%.10Lf", v->ldblval);
		char *c = malloc(len + 1);
		sprintf(c, "%.10Lf", v->ldblval);
		// remove trailing zeros (and dot if necessary)
		while(--len && c[len] == '0');
		if(c[len] != '.') len++;
		c[len] = '\0';
		return c;
		}
	case VBOOL:
		return strdup(v->boolval == true ? "true" : "false");
	case VUNDEF:
	case VJSONARRAY:
	case VJSONOBJECT:
	case VARRAY:
		break;
	// no default (relying on compiler warnings for detecting missing switch values)
	}
	return NULL;
}


/******************************************************************************
 *
 * get string value from value
 * returns: the string pertaining to the value, NULL if the type does not
 * stores its native value in a string form
 ****************************************************************************/


static char* getstring(const VAL v){
	switch(v->type){
	case VSTRING:
		return v->strval;
	case VLDOUBLE:
	case VBOOL:
	case VUNDEF:
	case VARRAY:
	case VJSONARRAY:
	case VJSONOBJECT:
		break;
	// no default (relying on compiler warnings for detecting missing switch values)
	}
	return NULL;
}



/******************************************************************************
 *
 * converts a value to double
 * returns NAN on failure
 ****************************************************************************/

static long double toldbl(const VAL v){
	switch(v->type){
		case VBOOL:
			return v->boolval;
		case VLDOUBLE:
			return v->ldblval;
		case VUNDEF:
			return NAN;
		default:
			return mystrtold(getstring(v));
	}
	return NAN;
}

/******************************************************************************
 *
 * converts a value to bool
 *
 ****************************************************************************/

static bool tobool(const VAL v){
	switch(v->type){
		case VBOOL:
			return v->boolval;
		case VLDOUBLE:
			return v->ldblval == 0. ? false : true;
		case VUNDEF:
			return false;
		case VARRAY:
			return true;
		default:
		{
			long double d = mystrtold(getstring(v));
			if(isnan(d) || d == 0.) return false;
			break;
		}
	}
	return true;
}

/******************************************************************************
 *
 * outputs a json object with HTML headers to stdout
 * the object fields are read from the stack
 * input: nbparams: number of elements to read from the stack
 *
 *****************************************************************************/


static void sendjson(size_t nbparams){
	STRING *json = newlstring();
	// with libfcgi I/O operations get a significant overhead, so
	// it's beneficial to build the buffer and send it at once

#ifdef DEBUG
	struct timeval *moo = timerstart();
#endif
	buildjson(json, nbparams);
	if(config.socketpath){
		FCGX_PutS("Content-Type:application/json; charset=UTF-8\r\n\r\n", request.out);
		FCGX_PutS(lstringbuffer(json), request.out);
	}
	else{
		puts("Content-Type:application/json; charset=UTF-8\r\n\r\n");
		puts(lstringbuffer(json));
	}
#ifdef DEBUG
	fflush(stdout);
	timer(moo);
#endif
	freelstring(json);
}

/******************************************************************************
 *
 * builds json object from stack
 * the object fields are read from the stack
 * input: nbparams: number of elements to read from the stack
 * returns: a JSON object
 *
 *****************************************************************************/



static void buildjson(STRING * const json, size_t nbparams){
	// used several json library here, dissatisfied with performance
	// (even "the fastest in the world", three times slower with 20kb json)
	// realized it was pointless and using a hammer to kill a fly
	char *name;
	lstringappendc(json, '{');

	while(nbparams){
		name = getstring(stackptr);
		if(!name){
			errprint("json key is not a string");
			name = ERRORJSON;
		}

		lstringappendc(json, '"');
		lstringappend(json, name);
		pop();
		lstringappend(json, "\":");
		switch(stackptr->type){
		case VJSONOBJECT:
		{
			size_t nparams = stackptr->jsonval.jsonsize;
			pop();
			buildjson(json, nparams);
			nbparams--;
			if(nbparams) lstringappendc(json, ',');
			continue;
		 }

		case VJSONARRAY:
			nbparams--;
			buildarray(json);
			if(nbparams) lstringappendc(json, ',');
			continue;
		case VBOOL:

			lstringappend(json, stackptr->boolval ? "true": "false");
			break;
		case VLDOUBLE:
		{
			char *c;
			if((long long)stackptr->ldblval == stackptr->ldblval)
				c = lsprintf("%lld", (long long)stackptr->ldblval);
			else c = lsprintf("%Lf", stackptr->ldblval);

			lstringappend(json,  c);
			free(c);
			break;
		}
		default:
			{
				const char *c = getstring(stackptr);

				if(!c) lstringappend(json, "null");
				else {
					lstringappendc(json, '"');
					lstringappend(json, c);
					lstringappendc(json, '"');
				}
			}
		}
		nbparams--;
		pop();

		if(nbparams) lstringappendc(json, ',');
	}
	lstringappendc(json, '}');
}


/******************************************************************************
 *
 * builds json array from query result
 * the fields are read from the stack (see code)
 * params: json: the json string being built
 * returns: a JON array
 *****************************************************************************/


static void buildarray(STRING * const json){
	size_t nbparams;
	struct col *cols;
	ht *results;

	if(stackptr->jsonval.results->type != VARRAY){
		errprint("not an array");
		lstringappend(json, ERRORJSON);
		return;
	}
	results = stackptr->jsonval.results->hash;
	nbparams = stackptr->jsonval.jsonsize; // number of fields to read from the stack
	if(!results) {
		nbparams = nbparams * 2 + 1;
		for(size_t i = 0; i < nbparams; i++) pop();
		lstringappend(json, "null");
		return;
	}
	pop();
	cols = malloc(nbparams * sizeof(struct col));

	for(size_t i = 0; i < nbparams; i++){
		cols[i].key = tostring(stackptr);
		if(!cols[i].key) {
			errprint("specified json key is not a string");
		}
		pop();
		cols[i].index = tostring(stackptr);
		if(!cols[i].index){
			errprint("specified json index is not a string");
		}
		pop();
	}

	lstringappendc(json, '[');
	long double row = 0;
	VAL currentval;
	bool first = true;

	while((currentval = ht_get(results, &row, 10, NULL))){
		if(first == false) lstringappendc(json, ',');
		else first = false;
		if(currentval->type != VARRAY){
			errprint("not a two dimensional array");
			continue;
		}
		lstringappendc(json, '{');
		for(size_t numcol = 0; numcol < nbparams; numcol++){
			char *index = cols[numcol].index;

			lstringappendc(json, '"');
			lstringappend(json, cols[numcol].key);
			lstringappend(json, "\":");


			if(index == NULL) {
				lstringappendc(json, '"');
				lstringappend(json, ERRORJSON);
				lstringappendc(json, '"');
				if(numcol < nbparams - 1) lstringappendc(json, ',');
				continue;
			}
			VAL column = ht_get(currentval->hash, index, strlen(index), NULL);

			if(column == NULL) lstringappend(json, "null");
			else if(column->type == VLDOUBLE){
				char *c = tostring(column);
				lstringappend(json, c);
				free(c);
			}
			else{
				lstringappendc(json, '"');
				lstringappend(json, column->strval);
				lstringappendc(json, '"');
			}
			if(numcol < nbparams - 1) lstringappendc(json, ',');
		}
		lstringappendc(json, '}');

		row++;
	}

	lstringappendc(json, ']');
	for(size_t i = 0; i < nbparams; i++) {
		if(cols[i].key) free(cols[i].key);
		if(cols[i].index) free(cols[i].index);
	}

	free(cols);
	//return array;
}



/******************************************************************************
 *
 * free a value
 *
 *****************************************************************************/


void freevarval(VAL v){
	switch(v->type){
		case VSTRING:
			free(v->strval);
		break;
		case VARRAY:
		{
			if(v->hash){
				hti* iterator = hti_create(v->hash);
				while(iterator->key){
					freevarval(&(iterator->current->value));
					hti_next(iterator);
				}
				free(iterator);
				
				ht_destroy(v->hash);
			}
			break;
		}
		case VBOOL:
		case VLDOUBLE:
		case VUNDEF:
		case VJSONARRAY:
		case VJSONOBJECT:
			break;
		// no default (relying on compiler warnings for detecting missing switch values)
	}
	v->type = VUNDEF;
}

#ifdef DEBUG
/******************************************************************************
 *
 * decode an instruction in readable format to stderr
 *
 *****************************************************************************/

static void displayinstr(INSTR instr){
	char *op1 = maddr(instr->opds.op1);
	char *op2 = maddr(instr->opds.op2);

	errout("%p  ", instr);
	switch(instr->op){
	case IEQUALS:
		errout("eq   %s, %s", op1, op2);
		break;
	case INEQUALS:
		errout("neq  %s, %s", op1, op2);
		break;
	case ISUPEQ:
		errout("ge   %s, %s", op1, op2);
		break;
	case ISUP:
		errout("g    %s, %s", op1, op2);
		break;
	case IINFEQ:
		errout("le   %s, %s", op1, op2);
		break;
	case IINF:
		errout("i    %s, %s", op1, op2);
		break;
	case IJZ:
		errout("jz   %s, %p", op1, instr->jump.jmpto);
		break;
	case IJMP:
		errout("jump %p", instr->jump.jmpto);
		break;
	case IASSIGN:
		errout("mov  %s, %s", op1, op2);
		break;
	case IPUSH:
		errout("push %p", instr->pushval);
		break;
	case IRETURN:
		errout("ret  %d", instr->pop);
		break;
	case IEXECQ:
		errout("exec %d", instr->pop);
		break;
	case IMILLIS:
		errout("getmillis()");
		break;
	case IUPPER:
		errout("upper(%s)", op1);
		break;
	case ILOWER:
		errout("lower(%s)", op1);
		break;
	case ISUBSTR:
		errout("substr()");
		break;
	case IPOST:
		errout("readpost()");
		break;
	case IADD:
		errout("add  %s, %s", op1, op2);
		break;
	case ISUB:
		errout("sub  %s, %s", op1, op2);
		break;
	case IMUL:
		errout("mul  %s, %s", op1, op2);
		break;
	case IDIV:
		errout("div  %s, %s", op1, op2);
		break;
	case ILOR:
		errout("lor  %s, %s", op1, op2);
		break;
	case ILAND:
		errout("land %s, %s", op1, op2);
		break;
	case IEND:
		errout("end");
		break;
	case IARR:
		errout("arr()");
		break;
	case IASSIGNARR:
		errout("mov arr()");
		break;
	case IDEBUG:
		errout("dbg()");
		break;
	default:
		for(int i = 0; i < sizeof(struct stinstruction); i++) errout("%02X ", ((char*)instr)[i] & 0xff);
	}
	free(op1);
	free(op2);
	terminatestderr();
}

/******************************************************************************
 *
 * returns allocated string containing address or register name
 *
 *****************************************************************************/

static char* maddr(VAL ptr){
	if(ptr == &ax) return lsprintf("%-14s", "ax");
	if(ptr == &bx) return lsprintf("%-14s", "bx");
	return lsprintf("%p", ptr);
}

/******************************************************************************
 *
 * returns allocated timeval struct containing current date
 *
 *****************************************************************************/


struct timeval* timerstart(void){
	struct timeval *now = malloc(sizeof(struct timeval));
	gettimeofday(now, NULL);
	return now;
}

/******************************************************************************
 *
 * prints to stderr the number of millisecondes between tm and current time
 *
 *****************************************************************************/

static void timer(struct timeval * const tm){
	struct timeval now, res;
	gettimeofday(&now, NULL);
	timersub(&now, tm, &res);
	free(tm);
	errprint("time: %.3fms", (double)res.tv_sec * 1000. + (double)res.tv_usec / 1000);
	terminatestderr();
}



#endif
