#include <string.h>
#include <fcgiapp.h>
#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>

#include "params.h"
#include "sql.h"
#include "main.h"
#include "utils.h"
#include "hash.h"

MYSQL *db;
extern struct stconfig config;
extern FCGX_Request request;

static CACHEDQUERY querycache;
static unsigned querycachesize;

static void sqlerror(void);
static CACHEDQUERY getquery(char*, size_t);
static void freebinds(MYSQL_BIND*, unsigned);
static void manageconnectionloss(void);
static void freequery(CACHEDQUERY);
static void closequerycache(CACHEDQUERY);
static void querycachedone(void);


/******************************************************************************
 *
 * executes a statemment, returns the result in the form of an hashtable
 * the query either prepared and put in cache or retrieved from cache
 * params: ********************************************************************
 * query:     the query in the form of a string, eventually containing question
 *            marks for the prepared statements parameters
 * p:         the corresponding parameters
 * numparams: number of parameters
 * output: ********************************************************************
 * ierror: 0 if ok, 1 if error
 * returns: *******************************************************************
 * NULL if no results, an hashtable of results otherwise
 * the results in the hashtables are stored as VAL values, either of type
 * VSTRING or VLDOUBLE depending on the numeric attribute of the retrieved
 * column.
 * the hashtable is two dimensional (hashtable of hashtables), the first layer
 * is indexed by row number (starting from zero), the second one by column name
 * (which are retrieved so "select *" will work)
 *
 *****************************************************************************/



ht* vquery(char query[], char *p[], size_t numparams, int *ierror){
	char *buf = cleanquery(query);

	*ierror = 1;
#ifdef DEBUG
	errprint("APPEL SQL \"%s\"", buf);
#endif

	CACHEDQUERY vquery = getquery(buf, numparams);

	if(vquery == NULL) return NULL;

	if(numparams){
		for(unsigned i = 0; i < numparams; i++){
			vquery->inlengths[i] = strlen(p[i]);
			vquery->inbinds[i].buffer = p[i];
			vquery->inbinds[i].buffer_length = vquery->inlengths[i];
			vquery->inbinds[i].is_null = 0;
			vquery->inbinds[i].length = vquery->inlengths + i;
		}
	}
	if(mysql_stmt_bind_param(vquery->stmt, vquery->inbinds)){
		sqlerror();
		querycachedone();

		return NULL;
	}

	if(mysql_stmt_execute(vquery->stmt)){
		sqlerror();
		querycachedone();
		return NULL;
	}

	// insert, update...
	if(!vquery->fieldcount){
		*ierror = 0;
		querycachedone();

		return NULL;
	}

	if(mysql_stmt_store_result(vquery->stmt)){
		sqlerror();
		querycachedone();
		return NULL;
	}
	my_ulonglong numrows = mysql_stmt_num_rows(vquery->stmt);
	if(!numrows) {
		*ierror = 0;
		querycachedone();
		return NULL;
	}
	MYSQL_RES *metadata = mysql_stmt_result_metadata(vquery->stmt);
	if(!metadata){
		sqlerror();
		querycachedone();
		return NULL;
	}

	MYSQL_FIELD *fields =  mysql_fetch_fields(metadata);
	if(!fields){
		sqlerror();
		mysql_free_result(metadata);
		querycachedone();
		return NULL;
	}
	mysql_free_result(metadata);

	MYSQL_BIND *bindouts = calloc(vquery->fieldcount, sizeof(MYSQL_BIND));
	struct morebindfields *morebinds = calloc(vquery->fieldcount, sizeof(struct morebindfields));

	for(unsigned i = 0; i < vquery->fieldcount; i++){
		bindouts[i].buffer_type = MYSQL_TYPE_STRING;
		bindouts[i].buffer = malloc(fields[i].max_length);
		bindouts[i].buffer_length = fields[i].max_length;
		bindouts[i].is_null = &(morebinds[i].is_null);
		bindouts[i].error = &(morebinds[i].error);
		bindouts[i].length = &(morebinds[i].length);
	}
	if(mysql_stmt_bind_result(vquery->stmt, bindouts)){
		freebinds(bindouts, vquery->fieldcount);
		free(morebinds);
		sqlerror();
		querycachedone();
		return NULL;
	}

	QRET queryresult = malloc(sizeof(struct stqret));
	queryresult->c = vquery->fieldcount;
	queryresult->namelens = malloc(vquery->fieldcount * sizeof(size_t));
	queryresult->names = malloc(vquery->fieldcount * sizeof(char*));
	queryresult->isnum = malloc(vquery->fieldcount * sizeof(char));


	for(unsigned i = 0; i < vquery->fieldcount; i++) {
		queryresult->namelens[i] = strlen(fields[i].name);
		queryresult->names[i] = malloc(queryresult->namelens[i] + 1);
		memcpy(queryresult->names[i], fields[i].name, queryresult->namelens[i] + 1);
		queryresult->isnum[i] = fields[i].flags & NUM_FLAG ? 1 : 0;
	/*	(fields[i].type == MYSQL_TYPE_TINY ||
		fields[i].type == MYSQL_TYPE_SHORT ||
		fields[i].type == MYSQL_TYPE_LONG ||
		fields[i].type == MYSQL_TYPE_INT24 ||
		fields[i].type == MYSQL_TYPE_LONGLONG ||
		fields[i].type == MYSQL_TYPE_DECIMAL ||
		fields[i].type == MYSQL_TYPE_NEWDECIMAL)
		? 1 : 0;*/
	}

	my_ulonglong row = 0;

	ht* results = ht_create(DEFAULT_HASH_SIZE);

	struct stvalue value, valueline;
	value.type = VSTRING;
	valueline.type = VARRAY;

	while (1){
		int status = mysql_stmt_fetch(vquery->stmt);
		if (status == 1){
			sqlerror();
			freequeryresult(queryresult);
			freebinds(bindouts, vquery->fieldcount);
			free(morebinds);
			querycachedone();
			return NULL;
		}

		if(status == MYSQL_NO_DATA) break;
		valueline.hash = ht_create(DEFAULT_HASH_SIZE);
		for(unsigned i = 0; i < vquery->fieldcount; i++){
			if(!morebinds[i].is_null){
				if(queryresult->isnum[i]){
					value.type = VLDOUBLE;
					value.ldblval = mystrtold(bindouts[i].buffer);
					ht_set(valueline.hash, queryresult->names[i], queryresult->namelens[i], &value);
				}
				else{
					value.type = VSTRING;

					value.strval = malloc(morebinds[i].length + 1);
					memcpy(value.strval, bindouts[i].buffer, morebinds[i].length);
					value.strval[morebinds[i].length] = '\0';
					ht_set(valueline.hash, queryresult->names[i], queryresult->namelens[i], &value);
				}
			}
		}
		long double frow = row;
		ht_set(results, &frow, 10, &valueline);
		row++;
	}
	freebinds(bindouts, vquery->fieldcount);
	free(morebinds);

	freequeryresult(queryresult);
	*ierror = 0;
	querycachedone();

	return results;

}


/******************************************************************************
 *
 * retrieves a query from cache. If found, the query is put first in the cache
 * else it is added, and if the query cache is full, the last one is discarded
 * params: ********************************************************************
 * query:     the query in the form of a string, eventually containing question
 *            marks for the prepared statements parameters
 * numparams: number of parameters (the binding allocation is done at this stage)
 * returns: *******************************************************************
 * NULL on error, a query cache structure otherwise
 *
 *****************************************************************************/



static CACHEDQUERY getquery(char *query, size_t numparams){
	CACHEDQUERY curr = querycache,
		old = querycache; // asssignment has no purpose, warning avoidance;
	const bool vtrue = true;

	manageconnectionloss();

	curr = querycache;

	while(curr && strcmp(query, curr->query)){
		old = curr;
		curr = curr->next;
	}
	// cache hit
	if(curr){
		free(query);
		if(curr != querycache){
			// move to first of list (LRU management)
			curr->prev->next = curr->next;
			if(curr->next) curr->next->prev = curr->prev;
			curr->next = querycache;
			querycache->prev = curr;
			curr->prev = NULL;
			querycache = curr;
		}
		return curr;
	}
	// cache miss
	if(querycachesize < config.querycachesize || config.querycachesize == 0){
		curr = calloc(1, sizeof(struct stcachedquery));
		querycachesize++;
	}
	else{
		// old is the last element of the list, we well use it
		curr = old;
		if(curr == querycache) querycache = NULL;
		if(curr->prev) curr->prev->next = NULL;
		closequerycache(curr);
		memset(curr, 0, sizeof(struct stcachedquery));
	}

	curr->stmt = mysql_stmt_init(db);
	curr->query = query;
	mysql_stmt_attr_set(curr->stmt, STMT_ATTR_UPDATE_MAX_LENGTH, &vtrue);

	if(mysql_stmt_prepare(curr->stmt, query, strlen(query))){
		sqlerror();
		freequery(curr);
		return NULL;
	}
	curr->fieldcount = mysql_stmt_field_count(curr->stmt);
	if(mysql_stmt_param_count(curr->stmt) != numparams){
		errprint("wrong number of query parameters for \"%s\"", curr->query);
		freequery(curr);
		return NULL;
	}
	if(numparams){
		curr->inbinds = calloc(numparams, sizeof(MYSQL_BIND));
		curr->inlengths = malloc(numparams * sizeof(unsigned long));
		for(size_t i = 0; i < numparams; i++){
			curr->inbinds[i].buffer_type = MYSQL_TYPE_STRING;
		}
	}
	curr->next = querycache;
	if(querycache) querycache->prev = curr;
	querycache = curr;

	return curr;
}

/******************************************************************************
 * frees the query when the cache is disabled by config
 *****************************************************************************/


static void querycachedone(void){
	if(config.querycachesize == 0){
		freequery(querycache);
		querycache = NULL;
	}
}


/******************************************************************************
 *
 * frees the binding buffers
 * params: ********************************************************************
 * b: the binding array address
 * l: the array size
 *
 *****************************************************************************/

static void freebinds(MYSQL_BIND *b, unsigned l){
	for(unsigned i = 0; i < l; i++) if(b[i].buffer) free(b[i].buffer);
	free(b);
}


/******************************************************************************
 *
 * purges the query cache
 *
 *****************************************************************************/


void freequerylist(){
	while(querycache){
		const CACHEDQUERY curr = querycache->next;
		freequery(querycache);
		querycache = curr;
	}
	querycachesize = 0;
}

/******************************************************************************
 *
 * frees a query element from the query cache
 * params: ********************************************************************
 * query:     a cached query element to free
 *
 *****************************************************************************/


static void freequery(CACHEDQUERY query){
	closequerycache(query);
	free(query);

}

/******************************************************************************
 *
 * close the prepared statement from the cached query and frees the mysql
 * associated buffers
 * params: ********************************************************************
 * query:     a cached query element
 *
 *****************************************************************************/


static void closequerycache(CACHEDQUERY query){
	if(query->stmt) mysql_stmt_close(query->stmt);
	if(query->query) free(query->query);
	if(query->inbinds) free(query->inbinds);
	if(query->inlengths) free(query->inlengths);
}


/******************************************************************************
 *
 * frees a QRET structure
 * params: ********************************************************************
 * res: the result structure
 *
 *****************************************************************************/


void freequeryresult(QRET res){
	for(unsigned i = 0; i < res->c; i++) free(res->names[i]);
	free(res->names);
	free(res->namelens);
	free(res->isnum);

	free(res);
}

/******************************************************************************
 *
 * checks if database connection was lost, and restores it if this is the case
 *
 *****************************************************************************/


static void manageconnectionloss(){
	if(mysql_ping(db)){
		db_disconnect();
		db_connect(0);
	}
}


/******************************************************************************
 *
 * performs a database connection
 * params: ********************************************************************
 * startup_connection: if put to 1, error_messages will be displayed on stderr
 *                     else they will be put tu the fcgi err channel
 * returns: *******************************************************************
 * 0 on success, 1 on error
 *
 *****************************************************************************/

int db_connect(int startup_connection){
	if(db) return 0;

	db = mysql_init(NULL);
	if(db == NULL) return 1;
	unsigned ssl_mode = SSL_MODE_DISABLED;
	mysql_options(db, MYSQL_OPT_SSL_MODE, &ssl_mode);


	if(mysql_real_connect(db, config.dbhost, config.dbuser, config.dbpass, config.dbname, config.dbport, config.dbsocket, CLIENT_MULTI_STATEMENTS) == NULL){
		if(startup_connection == 1 || !config.socketpath) fprintf(stderr, "Could not connect to MySQL (%d) - %s.\n", mysql_errno(db), mysql_error(db));
		else FCGX_FPrintF(request.err, "Could not connect to MySQL (%d) - %s.\n", mysql_errno(db), mysql_error(db));
		mysql_close(db);
		return 1;
	}
	if(mysql_set_character_set(db, "utf8")){
		if(startup_connection == 1 || !config.socketpath) fprintf(stderr, "Could not set datablase charset (%d) - %s.\n", mysql_errno(db), mysql_error(db));
		else FCGX_FPrintF(request.err, "Could not set datablase charset (%d) - %s.\n", mysql_errno(db), mysql_error(db));
		mysql_close(db);
		return 1;
	}
	return 0;
}

/******************************************************************************
 *
 * set autocommit on or off
 * params: ********************************************************************
 * v: true for autocommit, false otherwise
 * returns: *******************************************************************
 * 1 on error, zero on success
 *
 *****************************************************************************/


int setautocommit(bool v){
	if(mysql_autocommit(db, v) != 0){
		sqlerror();
		return 1;
	}
	return 0;
}


/******************************************************************************
 *
 * disconnects from database and purge prepared statements cache
 *
 *****************************************************************************/



void db_disconnect(void){
	mysql_close(db);
	mysql_library_end();
	freequerylist();
	db = NULL;
}


/******************************************************************************
 *
 * displays mysql error to error channel
 *
 *****************************************************************************/


static void sqlerror(void){
	errprint("MySQL error (%d) - %s\n", mysql_errno(db), mysql_error(db));
}
