#include <stdio.h>
extern FILE *yyin;
#include <fcgiapp.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>
#include <regex.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <sys/stat.h>
#include <errno.h>

#include "sql.h"
#include "params.h"
#include "vm.h"
#include "utils.h"
#include "main.h"
#include "confreader.h"
#include "cpufreq.h"


int yyparse();
static void decodeHeader(const char*);
static void readposts(void);
static void freeposts(void);
static int parseconfig(const char path[]);
static int strtounsigned(const char*, unsigned* const);
static int managesigterm(void);
static int fcgistartup(void);
static int fcgiaccept(void);

extern VAR vars;        // variables used by script
struct pargs *post;     // linked list of post key/values
static regex_t preg;    // precompiled regex for http header post values
struct stconfig config; // structure storing configuration for config file
extern int yydebug;
FCGX_Request request;





int main(int argc, char *argv[]){
	yydebug = 0;

	if(argc != 2){
		fprintf(stderr, "usage: machin_fcgi config_file\n");
		return 1;
	}

	if(parseconfig(argv[1])) return EXIT_FAILURE;
	// script file compilation
	FILE *f = fopen(config.progpath, "r");
	if(f == NULL){
		fprintf(stderr, "Cannot open script file: %s\n", strerror(errno));
		return EXIT_FAILURE;
	}
	yyin = f;
	if(yyparse()){
		fclose(f);
		return EXIT_FAILURE;
	}
	fclose(f);
	
	regcomp(&preg, "Content-Disposition: form-data; name=\"(.*)\"[\r\n]+([^\r\n]*)[\r\n]+", REG_EXTENDED | REG_NEWLINE);

	if(db_connect(1)) return 1;
	initvm();
	initgovernor(config.governoractivetime, config.idlegovernor, config.execgovernor);
	managesigterm();

	if(fcgistartup()) return EXIT_FAILURE;

	// main loop
	while(fcgiaccept() >= 0){
		gouptime();
		// put it here so to allow thread-safe in later improvement
		// might become a whole struct if isages like this grow
		struct timeval tm;
		gettimeofday(&tm, NULL);
		readposts();
		runVM(&tm);
		if(config.socketpath) FCGX_Finish_r(&request);
		else fflush(stdout);
		freeposts();
		for(VAR curr = vars; curr; curr = curr->next) freevarval(curr->val);
	}
	freequerylist();
	db_disconnect();
	if(config.socketpath) FCGX_Free(&request, 1);

	regfree(&preg);

	return EXIT_SUCCESS;
}


/******************************************************************************
 *
 * wrapper for FCGX_Accept allowing a CGI behaviour if the fcgi socket is
 * not set. In that case, the second call will invariably fail
 * NOT THREAD SAFE
 * returns: *******************************************************************
 * -1 on error, 0 on success
 *
 *****************************************************************************/

static int fcgiaccept(void){
	static int ran = 0;
	if(config.socketpath) return FCGX_Accept_r(&request);
	if(ran != 0) return -1;
	ran++;
	return 0;
}

/******************************************************************************
 *
 * initialize FCGX if the fcgi socket is set in config
 * returns: *******************************************************************
 * 1 on error, 0 on success
 *
 *****************************************************************************/


static int fcgistartup(void){
	if(!config.socketpath) return 0;
	FCGX_Init();
	mode_t mask = umask(0);
	int socket = FCGX_OpenSocket(config.socketpath, SOCKET_BACKLOG);
	umask(mask);
	if(socket == -1){
		fprintf(stderr, "could not open socket \"%s\": %s", config.socketpath, strerror(errno));
		return 1;
	}
	FCGX_InitRequest(&request, socket, 0);

	return 0;
}


/******************************************************************************
 *
 * reads the in channel into an allocated buffer, decodes http post values
 * and put them in a post structure linked list
 *
 *****************************************************************************/


static void readposts(void){
	char *buf, *ptr;
	size_t readsize = POST_BUF_LEN;
	size_t mallocsize = POST_BUF_LEN;
	size_t breadtotal = 0;

	buf = malloc(mallocsize);
	ptr = buf;

	while(1){
		size_t bread;
		if(config.socketpath) bread = FCGX_GetStr(ptr, readsize - 1, request.in);
		else bread = fread(ptr, 1, readsize - 1, stdin);
		if(bread < readsize - 1) {
			ptr[bread] = '\0';
			break;
		}
		breadtotal += bread;
		mallocsize *= 2;

		buf = realloc(buf, mallocsize);
		ptr = buf + breadtotal;

		readsize = mallocsize / 2;
	}
	decodeHeader(buf);
	free(buf);
}

/******************************************************************************
 *
 * frees the post linked list
 *
 *****************************************************************************/


static void freeposts(void){
	struct pargs *next;
	while(post){
		next = post->next;
		free(post->key);
		free(post->val);
		free(post);
		post = next;
	}
	post = NULL;
}


/******************************************************************************
 *
 * extracts post key/value paris from http headers and put them in the post
 * linked list
 * params: ********************************************************************
 * hdr:     a string containing the raw http headers
 *
 *****************************************************************************/


static void decodeHeader(const char *hdr){
	regmatch_t pmatch[3];
	while(1){
		if(regexec (&preg, hdr, 3, pmatch, 0) != 0) break;

		struct pargs *newpost = malloc(sizeof(struct pargs));

		newpost->key = strndup(hdr + pmatch[1].rm_so, pmatch[1].rm_eo - pmatch[1].rm_so);
		newpost->val = strndup(hdr + pmatch[2].rm_so, pmatch[2].rm_eo - pmatch[2].rm_so);
		newpost->next = post;
		post = newpost;
		hdr += pmatch[0].rm_eo;
	}
}

/******************************************************************************
 *
 * restores the cpufreq governor on SIGTERM ans SIGINT
 * returns: *******************************************************************
 * 0
 *****************************************************************************/

static int managesigterm(void){
	struct sigaction action;
	memset(&action, 0, sizeof(action));
	action.sa_handler = stopgovernor;
	sigaction(SIGTERM, &action, NULL);
	sigaction(SIGINT, &action, NULL);
	return 0;
}


/******************************************************************************
 *
 * fills the config structure from config file
 * params: ********************************************************************
 * path: the config file path
 * returns: *******************************************************************
 * 1 on error, zero otherwise
 *
 *****************************************************************************/




static int parseconfig(const char path[]){
	CONF conf = confreader(path);
	if(!conf) return 1;
	CONF node = conf;
	int ret = 0;
	int querycachesizeset = false;
	config.dbport = 0;

	while(node){
		if(node->val[0] == '\0') {
			node = node->next;
			continue;
		}
		if(!strcmp(node->key, "db_user")) config.dbuser = strdup(node->val);
		else if(!strcmp(node->key, "db_password")) config.dbpass = strdup(node->val);
		else if(!strcmp(node->key, "db_name")) config.dbname = strdup(node->val);
		else if(!strcmp(node->key, "db_host")) config.dbhost = strdup(node->val);
		else if(!strcmp(node->key, "db_port")){
			int tmp = strtounsigned(node->val, &config.dbport);
			if(tmp < 0 || config.dbport > 65535){
				fprintf(stderr, "Invalid value for option \"%s\".\n", node->key);
				ret = 1;
			}
		}
		else if(!strcmp(node->key, "db_socket")) config.dbsocket = strdup(node->val);
		else if(!strcmp(node->key, "program")) config.progpath = strdup(node->val);
		else if(!strcmp(node->key, "fcgi_socket")) config.socketpath = strdup(node->val);
		else if(!strcmp(node->key, "execution_governor")) config.execgovernor = strdup(node->val);
		else if(!strcmp(node->key, "idle_governor")) config.idlegovernor = strdup(node->val);
		else if(!strcmp(node->key, "execution_governor_duration")){
			int tmp = strtounsigned(node->val, &config.governoractivetime);
			if(tmp < 0){
				fprintf(stderr, "Invalid value for option \"%s\". Ignored.\n", node->key);
				ret = 1;
			}
		}
		else if(!strcmp(node->key, "query_cache_size")){
			int tmp = strtounsigned(node->val, &config.querycachesize);
			if(tmp < 0){
				fprintf(stderr, "Invalid value for option \"%s\". Ignored.\n", node->key);
				ret = 1;
			}
			querycachesizeset = true;
		}
		else fprintf(stderr, "Unknown config option \"%s\". Ignored.\n", node->key);

		node = node->next;
	}
	if(config.governoractivetime > 1 && (config.execgovernor == NULL || config.idlegovernor == NULL)){
		fprintf(stderr, "missing governor(s)");
		ret = 1;
	}
	if(!config.progpath || !config.progpath[0]){
		fprintf(stderr, "program path not set in config file");
		ret = 1;
	}
	if(querycachesizeset == false){
		fprintf(stderr, "query_cache_size not set, defaulting to %d.\n", DEFAULT_QUERY_CACHE_SIZE);
		config.querycachesize = DEFAULT_QUERY_CACHE_SIZE;
	}
	if(config.dbuser == NULL){
		fprintf(stderr, "db_user not defined in config file. Aborting\n");
		ret = 1;
	}
	if(config.dbname == NULL){
		fprintf(stderr, "db_name not defined in config file. Aborting\n");
		ret = 1;
	}
	freeconf(conf);
	return ret;
}




/******************************************************************************
 *
 * wrapper for strtoul
 * params: ********************************************************************
 * string: a string representing an array
 * output: ********************************************************************
 * result: the decoded unsigned, will be untouched on error
 * returns: *******************************************************************
 * -1 on error, zero otherwise
 *
 *****************************************************************************/



static int strtounsigned(const char *string, unsigned * const result){
	if(string == NULL) return -1;
	char *end;
	unsigned long ul = strtoul(string, &end, 10);
	if(end == string || *end != '\0' || errno == ERANGE)
		return -1;

	*result = ul;
	return 0;
}



