%{
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <regex.h>
#include "blang.h"
#include "utils.h"

/*
	TODO: undeclared variable detection
*/

int yylex();
int yyerror(char *s);
static int error(char*, ...);
static INSTR newinstr(const enum opcode);
static void appendinstr(struct stblock* const, INSTR const);
static void append(struct stblock* const, struct stblock* const);
static VAL newval(const enum val);
static struct steval* createunaryexpr(const int, struct steval*);
static struct steval* createexpr(const int, struct steval*, struct steval*);
static struct stblock* createvariadicfunc(enum opcode, struct stblock*);
static void bxmov(struct steval*, struct steval*);
static VAR getvar(const char[], const bool);
extern int yylineno;
VAR vars;
struct stblock program;
struct stvalue ax, bx;

%}
%locations
%union{
  char *str;
  int ubool;
  struct stinstruction *instr;
  struct stblock *block;
  struct stvalue *value;
  struct steval *eval;
  long double ldbl;
  struct arrayindex *arrindex;
  int op;
}


%token             COLON RETURN COMMA PARENL PARENR BRACKETL BRACKETR CURLYL CURLYR SQL MILLIS LOG
%token             UPPER LOWER SUBSTR TOSTDERR UNDEFINED
%token             IF FI ELSE WHILE WEND
%token <str>       IDENTIFIER STRING POST
%token <ldbl>      LDOUBLE
%token <ubool>     BOOL
%left  <op>        LOR LAND
%left  <op>        DOUBLEEQUAL NOTEQUAL
%left  <op>        EQUAL SUPEQ INFEQ INF SUP
%left  <op>        PLUS MINUS
%left  <op>        MUL DIV
%left  <op>        NOT
%type <block>      jsonexpr jsonarrayfield jsonarrayfields jsonfieldlist jsonvalue negvalue array postval
%type <block>      return sql debug millis returnvalue params stmts param block assign if while substr arrayindex
%type <eval>       expr
%type <value>      value
%%


program: stmts{
	while(vars){
		VAR next = vars->next;
		free(vars->name);
		free(vars);
		vars = next;
	}
	program = *$1;
	free($1);
}


stmts:  block
| stmts block{
	append($1, $2);
}


negvalue: MINUS IDENTIFIER{
	VAR v = getvar($2, true);
	if(!v) return 1;
	free($2);
	$$ = calloc(1, sizeof(struct stblock));
	INSTR instr = newinstr(INEG);
	instr->opds.op1 = v->val;
	appendinstr($$, instr);
}|MINUS returnvalue{
	$$ = $2;
	INSTR instr = newinstr(INEG);
	instr->opds.op1 = &ax;
	appendinstr($$, instr);
}


arrayindex: BRACKETL param BRACKETR{
	$$ = $2;
	$$->nbparams = 1;
}
| arrayindex BRACKETL param BRACKETR{
	$$ = $3;
	$3->nbparams = $1->nbparams + 1;
	append($$, $1);
}








value: BOOL{
	$$ = newval(VBOOL);
	$$->boolval = $1;
}|IDENTIFIER{
	VAR v = getvar($1, false);
	if(!v) {
		v = calloc(1, sizeof(struct var));
		v->val = newval(VUNDEF);
		v->name = $1;
		v->next = vars;
		vars = v;
	}
	else free($1);
	$$ = v->val;
}
|STRING{
	$$ = newval(VSTRING);
	$$->strval = $1;
}|LDOUBLE{
	$$ = newval(VLDOUBLE);
	$$->ldblval = $1;
}|UNDEFINED{
	$$ = newval(VUNDEF);
}|MINUS LDOUBLE{
	$$ = newval(VLDOUBLE);
	$$->ldblval = -$2;
}

block:  sql | debug | millis | return | assign | if | while | substr | array | postval


while: WHILE expr stmts WEND{
	$$ = $2->block;

	INSTR instr = newinstr(IJMP);
	instr->jump.jmpto = $$->first;
	appendinstr($3, instr);

	instr = newinstr(IJZ);
	instr->jump.testval = $2->value;
	instr->jump.jmpto   = $3->last;
	free($2);
	appendinstr($$, instr);

	append($$, $3);
}

if: IF expr stmts FI{
	$$ = $2->block;
	INSTR instr = newinstr(IJZ);
	instr->jump.testval = $2->value;
	instr->jump.jmpto   = $3->last;
	free($2);
	appendinstr($$, instr);
	append($$, $3);
}| IF expr stmts ELSE stmts FI{
	$$ = $2->block;
	INSTR instr = newinstr(IJMP);
	instr->jump.jmpto = $5->last;
	appendinstr($3, instr);

	instr = newinstr(IJZ);
	instr->jump.testval = $2->value;
	instr->jump.jmpto   = $3->last;
	free($2);
	appendinstr($$, instr);
	append($$, $3);
	append($$, $5);
}

expr: PARENL expr PARENR{
	$$ = $2;
}|expr LAND expr{
	$$ = createexpr(ILAND, $1, $3);
}|expr LOR expr{
	$$ = createexpr(ILOR, $1, $3);
}|expr DOUBLEEQUAL expr{
	$$ = createexpr(IEQUALS, $1, $3);
}|expr NOTEQUAL expr{
	$$ = createexpr(INEQUALS, $1, $3);
}|expr SUPEQ expr{
	$$ = createexpr(ISUPEQ, $1, $3);
}|expr SUP expr{
	$$ = createexpr(ISUP, $1, $3);
}|expr INFEQ expr{
	$$ = createexpr(IINFEQ, $1, $3);
}|expr INF expr{
	$$ = createexpr(IINF, $1, $3);
}|expr MUL expr{
	$$ = createexpr(IMUL, $1, $3);
}|expr DIV expr{
	$$ = createexpr(IDIV, $1, $3);
}|expr PLUS expr{
	$$ = createexpr(IADD, $1, $3);
}|expr MINUS expr{
	$$ = createexpr(ISUB, $1, $3);
}|NOT expr{
	$$ = createunaryexpr(INOT, $2);
}| UPPER PARENL expr PARENR{
	$$ = createunaryexpr(IUPPER, $3);
}| LOWER PARENL expr PARENR{
	$$ = createunaryexpr(ILOWER, $3);
}|value{
	$$ = calloc(1, sizeof(struct steval));
	$$->block = calloc(1, sizeof(struct stblock));
	$$->value = $1;
}|negvalue{
	$$ = calloc(1, sizeof(struct steval));
	$$->block = $1;
	$$->value = &ax;
}|returnvalue{
	$$ = calloc(1, sizeof(struct steval));
	$$->block = $1;
	$$->value = &ax;
}

// functions which return a value
returnvalue: sql | millis | substr | array | postval

params: param {
	$$->nbparams = 1;
} | param COMMA params {
	$$ = $3;
	append($$, $1);
	$$->nbparams++;
}




param : expr{
	if(!$1->block) $1->block = calloc(1, sizeof(struct stblock));
	$$ = $1->block;
	INSTR instr = newinstr(IPUSH);
	instr->pushval = $1->value;
	appendinstr($$, instr);
	free($1);
}


assign: IDENTIFIER EQUAL expr{
	VAR v = getvar($1, false);
	if(!v) {
		v = calloc(1, sizeof(struct var));
		v->val = newval(VUNDEF);
		v->name = $1;
		v->next = vars;
		vars = v;
	}
	else free($1);


	INSTR instr = newinstr(IASSIGN);
	instr->opds.op1 = v->val;
	instr->opds.op2 = $3->value;

	if($3->block == NULL) $$ = calloc(1, sizeof(struct stblock));
	else $$ = $3->block;
	free($3);

	appendinstr($$, instr);
}| IDENTIFIER arrayindex EQUAL expr{
	VAR v = getvar($1, false);
	if(!v) {
		v = calloc(1, sizeof(struct var));
		v->val = newval(VUNDEF);
		v->name = $1;
		v->next = vars;
		vars = v;
	}
	else free($1);
	$$ = calloc(1, sizeof(struct stblock));
	INSTR instr = newinstr(IPUSH);
	instr->pushval = $4->value;
	appendinstr($$, instr);
	$$->nbparams = 1;

	instr = newinstr(IASSIGNARR);
	instr->array.pop = $2->nbparams + 1;
	instr->array.array = v->val;

	append($$, $2);

	if($4->block) append($$, $4->block);
	free($4);

	appendinstr($$, instr);
}

// this is a function in disguise

array: IDENTIFIER arrayindex{
	VAR v = getvar($1, false);
	if(!v) return 1;
	free($1);

	$$ = $2;
	INSTR instr = newinstr(IARR);
	instr->array.pop = $2->nbparams;
	instr->array.array = v->val;
	appendinstr($$, instr);
}



debug: TOSTDERR PARENL params PARENR {
	$$ = createvariadicfunc(IDEBUG, $3);
}


sql: SQL PARENL params PARENR {
	$$ = createvariadicfunc(IEXECQ, $3);
}


millis: MILLIS PARENL PARENR{
	$$ = calloc(1, sizeof(struct stblock));
	INSTR instr = newinstr(IMILLIS);
	appendinstr($$, instr);
}



substr: SUBSTR PARENL param COMMA param COMMA param PARENR {
	$$ = $7;
	append($$, $5);
	append($$, $3);
	$$->nbparams += 2;
	INSTR instr = newinstr(ISUBSTR);
	appendinstr($$, instr);
}

postval:POST{
	$$ = calloc(1, sizeof(struct stblock));
	INSTR instr = newinstr(IPOST);
	instr->postname = $1; // put names in linked list to save mem?
	appendinstr($$, instr);
}




/**********************************************
*
*    JSON
*
**********************************************/


return: RETURN jsonfieldlist{
	$$ = $2;
	INSTR instr = newinstr(IRETURN);
	instr->pop = $2->nbparams;
	appendinstr($$, instr);
};

jsonfieldlist: jsonexpr {
	$$->nbparams = 1;
} | jsonexpr COMMA jsonfieldlist{
	$$ = $3;
	$$->nbparams++;
	append($$, $1);
}

jsonexpr: param COLON jsonvalue{
	$$ = $3;
	append($$, $1);
}

jsonvalue: IDENTIFIER BRACKETL jsonarrayfields BRACKETR {
	VAR v = getvar($1, true);
	if(!v) return 1;
	free($1);
	$$ = $3;
	INSTR instr = newinstr(IPUSH);
	instr->pushval = newval(VJSONARRAY);
	instr->pushval->jsonval.jsonsize = $$->nbparams;
	instr->pushval->jsonval.results = v->val;
	appendinstr($$, instr);
}| param
| CURLYL jsonfieldlist CURLYR {
	$$ = $2;
	INSTR instr = newinstr(IPUSH);
	instr->pushval = newval(VJSONOBJECT);
	instr->pushval->jsonval.jsonsize = $$->nbparams;
	appendinstr($$, instr);
}


jsonarrayfields: jsonarrayfield{
	$$->nbparams = 1;
} | jsonarrayfield COMMA jsonarrayfields{
	$$ = $3;
	append($$, $1);
	$$->nbparams++;
}

jsonarrayfield: param COLON param{
	$$ = $3;
	append($3, $1);
}


%%

static struct stblock* createvariadicfunc(enum opcode op, struct stblock *params){
	INSTR instr = newinstr(op);
	instr->pop = params->nbparams;
	appendinstr(params, instr);
	return params;
}

// generates one operands operation

static struct steval* createunaryexpr(const int op, struct steval *v1){
	if(v1->block == NULL) v1->block = calloc(1, sizeof(struct stblock));

	INSTR instr = newinstr(op);
	instr->opds.op1 = v1->value;
	appendinstr(v1->block, instr);
	v1->value = &ax;
	return v1;
}
// generates two operands operation

static struct steval* createexpr(const int op, struct steval *v1, struct steval *v2){
	if(v1->block == NULL) v1->block = calloc(1, sizeof(struct stblock));
	bxmov(v1, v2);
	if(v2->block) append(v1->block, v2->block);

	INSTR instr = newinstr(op);
	instr->opds.op1 = v1->value;
	instr->opds.op2 = v2->value;
	free(v2);
	appendinstr(v1->block, instr);
	v1->value = &ax;
	return v1;
}


static void bxmov(struct steval *v1, struct steval *v2){
	// both values are stored in ax, move one to bx (moving from ax has a fast implementaton in vm)
	if(v1->value == &ax && v2->value == &ax){
		INSTR instr = newinstr(IASSIGN);
		instr->opds.op1 = &bx;
		instr->opds.op2 = &ax;
		appendinstr(v1->block, instr);
		v1->value = &bx;
	}
}

static VAL newval(const enum val type){
	VAL const ret = calloc(1, sizeof(struct stvalue));
	ret->type = type;
	return ret;
}

static INSTR newinstr(const enum opcode op){
	INSTR const ret = calloc(1, sizeof(struct stinstruction));
	ret->op = op;
	ret->srcline = yylineno;
	return ret;
}


static int error(char *s, ...){
	va_list ap;

	va_start(ap, s);
	char *buf = lprintf(s, ap);
	va_end(ap);
	const int ret = yyerror(buf);
	free(buf);
	return ret;
}


static VAR getvar(const char name[], const bool disperr){
	VAR v = vars;

	while(v && strcmp(v->name, name)) v = v->next;
	if(v == NULL && disperr == true) error("variable \"%s\" not found line", name);
	return v;
}



// merge two blocks


static void append(struct stblock* const blockA, struct stblock* const blockB){
	if(blockB->first){
		if(blockA->last) {
			for(INSTR instr = blockB->first; instr; instr = instr->next){
				// relocate jumps
				if((instr->op == IJMP || instr->op == IJZ) && instr->jump.jmpto == blockB->first)
					instr->jump.jmpto = blockA->last;
			}
			*(blockA->last) = *(blockB->first);
			free(blockB->first);
		}
		else blockA->first = blockB->first;

		blockA->last = blockB->last;
	}
	free(blockB);
}



static void appendinstr(struct stblock * const block, INSTR const instr){
	if(block->last){
		// move the IEND instr to the end
		struct stinstruction iend;
		iend = *(block->last);
		*(block->last) = *instr;
		*instr = iend;
		block->last->next = instr;
		block->last = instr;
	}
	else {
		block->first = instr;
		INSTR const iend = newinstr(IEND);
		instr->next = iend;
		block->last = iend;
	}
}


int yyerror(char *s) {
	printf("%s line %d\n", s, yylineno);
	return 0;
}


