#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <regex.h>
#include <errno.h>
#include <ctype.h>

#include "confreader.h"

// kept as generic as possible

static void appendconf(CONF*, const char[], regmatch_t*);


CONF confreader(const char path[]){
	char buf[INI_LINE_MAX];
	unsigned line = 0;
	CONF ret = NULL;
	regmatch_t pmatch[3];
	regex_t pregval;
	regex_t pregspace;

	if(regcomp(&pregval, "^[ \t]*([^= \t;#]+)[ \t]*=[ \t]*([^=]*[^= \t])?[ \t]*\n?$", REG_EXTENDED | REG_NEWLINE)){
		fprintf(stderr, "Regex compile error\n");
		return ret;
	}
	if(regcomp(&pregspace, "^[ \t]*([#;].*)?\n?$", REG_EXTENDED | REG_NOSUB)){
		regfree(&pregval);
		fprintf(stderr, "Regex compile error\n");
		return ret;
	}


	FILE *f = fopen(path, "r");
	if(!f) {
		fprintf(stderr, "Unable to open config file \"%s\": %s\n", path, strerror(errno));
		regfree(&pregval);
		regfree(&pregspace);
		return NULL;
	}
	while(fgets(buf, INI_LINE_MAX, f)) {
		line++;
		if(strlen(buf) == INI_LINE_MAX - 1 && buf[INI_LINE_MAX - 2] != '\n'){
			fprintf(stderr, "Line %u too long in config file. Aborting", line);
			fclose(f);
			freeconf(ret);
			regfree(&pregval);
			regfree(&pregspace);
			return NULL;
		}
		// comment line
		if(!regexec(&pregspace, buf, 0, NULL, 0)) continue;
		if(!regexec(&pregval, buf, 3, pmatch, 0)) appendconf(&ret, buf, pmatch);
		else fprintf(stderr, "Malformed line %u in conf file: %s", line, buf);
	}
	regfree(&pregval);
	regfree(&pregspace);
	if(ferror(f)){
		fprintf(stderr, "Error reading config file \"%s\": %s\n", path, strerror(errno));
		freeconf(ret);
		ret = NULL;
	}
	fclose(f);

	return ret;
}


static void appendconf(CONF *conf, const char buf[], regmatch_t *match){
	CONF node = malloc(sizeof(struct conf));
	node->key = malloc(match[1].rm_eo - match[1].rm_so + 1);
	sprintf(node->key, "%.*s", match[1].rm_eo - match[1].rm_so, buf + match[1].rm_so);
	for(unsigned i = 0; node->key[i]; i++) node->key[i] = tolower(node->key[i]);
	node->val = malloc(match[2].rm_eo - match[2].rm_so + 1);
	sprintf(node->val, "%.*s", match[2].rm_eo - match[2].rm_so, buf + match[2].rm_so);
	node->next = *conf;
	*conf = node;
}



char *getconfval(CONF conf, const char key[]){
	CONF node;
	for(node = conf; node; node = node->next)
		if(!strcmp(key, node->key)) return node->val;

	return NULL;
}


void freeconf(CONF conf){
	CONF next;
	while(conf){
		next = conf->next;
		free(conf->key);
		free(conf->val);
		free(conf);
		conf = next;
	}
}

