//#include <stdio.h>
#include <string.h>
#include <stdint.h>
#include <stddef.h>
#include "hash.h"

// put here in order not to pollute namespace
#define FNV_OFFSET 14695981039346656037UL
#define FNV_PRIME 1099511628211UL
#define ENTRYINDEX(x, y, z) (ht_entry*)((uint8_t*)(x)+(y)*(z))
#define NEXTENTRY(x, y) (x)=(ht_entry*)((uint8_t*)(x)+(y))



static uint64_t hash_key(const char * restrict);
static int ht_expand(ht* const restrict);
static ht_entry* alloc_entries(const size_t, const size_t, ht_entry**);
static void* setentry(ht_entry * const restrict, const char* const restrict, const uint64_t, const void* const restrict, const size_t);
static ht_entry* findentriestorelocate(const ht* const restrict, const char* const restrict, size_t* const restrict);
static ht_entry* moveentries(const ht* const restrict, ht_entry*, const size_t);
static void insertentries(const ht* const restrict, const ht_entry* restrict, size_t);

/************************************************************
*************************************************************
"PUBLIC" functions
*************************************************************
************************************************************/




/************************************************************
ht* ht_create(size_t datasize, size_t capacity)
*************************************************************
Description: Allocates an hashtable
Params in:   datasize: the size of the values to store
             capacity: initial and minimal capacity of the
                       table (in number of elements)
Returns:     An hashtable, NULL if malloc failed
************************************************************/

ht* ht_create(const size_t datasize, const size_t capacity) {
	ht* const restrict hash = malloc(sizeof(ht));
	if (hash == NULL) return NULL;

	hash->length = 0;
	hash->capacity = hash->mincapacity = capacity;
	// memory alignment, we pad to sizeof(intmax_t)
	hash->entrysize = datasize + sizeof(intmax_t) - datasize % sizeof(intmax_t) +
		// compactness, compilers might add padding to end of struct, which we discard
		offsetof(ht_entry, value);
	hash->valuesize = datasize;

	hash->entries = alloc_entries(capacity, hash->entrysize, &(hash->lastentry));
	if (hash->entries == NULL) {
		free(hash);
		return NULL;
	}
	//hash->lastentry = ENTRYINDEX(table->entries, table->entrysize, table->capacity - 1);
	return hash;
}





/************************************************************
void ht_destroy(ht *hash)
*************************************************************
Description: Deallocates an hashtable
Params in:   hash: the hashtable to destroy
************************************************************/

void ht_destroy(ht* const restrict hash) {
	free(hash->entries);
	free(hash);
}





/************************************************************
void* ht_get(ht *table, char *key, void *defaultval)
*************************************************************
Description: Reads a value drom hashtable. If defaultval is
             not NULL and the value is not found, it is then
             created
Params in:   table: the hash table
             key: the key to read
             defaultval: see description
Returns:     the value associated with the key, or NULL if
             key not found and defaultval is NULL
************************************************************/

void* ht_get(ht* const restrict table, const char * const restrict key, const void * const restrict defaultval) {
	const uint64_t hash = hash_key(key);
	const size_t index = hash % table->capacity;
	ht_entry * restrict entry = ENTRYINDEX(table->entries, table->entrysize, index);

	while(entry->key[0] != '\0') {
		if (strcmp(key, entry->key) == 0) return entry->value;
		NEXTENTRY(entry, table->entrysize);
		if(entry > table->lastentry) entry = table->entries;
	}
	if(defaultval == NULL) return NULL;
	table->length++;
	return setentry(entry, key, hash, defaultval, table->valuesize);
}





/************************************************************
void* ht_set(ht* table, char *key, void *value)
*************************************************************
Description: creates a (kay, value) pair in the table
Params in:   table: the hash table
             key: the key to create
             value: the associated value
Returns:     value on success, NULL on allocation failure
************************************************************/

void* ht_set(ht* const restrict table, const char * const restrict key, const void * const restrict value){
	if(table->length > table->capacity / 2 && ht_expand(table)) return NULL;
	const uint64_t hash = hash_key(key);
	const size_t index = hash % table->capacity;
	ht_entry * restrict entry = ENTRYINDEX(table->entries, table->entrysize, index);

	while(entry->key[0] != '\0') {
		if (strcmp(key, entry->key) == 0){
			memcpy(entry->value, value, table->valuesize);
			return entry->value;
		}
		NEXTENTRY(entry, table->entrysize);
		if(entry > table->lastentry) entry = table->entries;
	}
	table->length++;
	return setentry(entry, key, hash, value, table->valuesize);
}





/************************************************************
int ht_unset(ht *table, char* key)
*************************************************************
Description: removes a (key, value) from the table
Params in:   table: the hash table
             key: the key to be deleted
Returns:     0 on success, 1 if key not found,
             -1 on allocation failure
************************************************************/

int ht_unset(ht* const restrict table, const char* const restrict key){

	if(table->capacity > table->mincapacity * 2 && table->length <= table->capacity / 4 && ht_expand(table)) return -1;  // ajouter une option pour ça dans le ht_create?

	size_t count;
	ht_entry *entrystart = findentriestorelocate(table, key, &count);
	if(entrystart == NULL) return 1;

	entrystart->key[0] = '\0';
	table->length--;

	if(count == 0) return 0;

	NEXTENTRY(entrystart, table->entrysize);
	if(entrystart > table->lastentry) entrystart = table->entries;
	
	ht_entry *buf = moveentries(table, entrystart, count);
	if(buf == NULL) return 1;

	insertentries(table, buf, count);

	free(buf);

	return 0;
}





/************************************************************
hti* hti_create(ht *table)
*************************************************************
Description: Allocates an iterator
Params in:   table: the hash table on which an iterator will
                    be built
Returns:     An iterator, NULL if malloc failed
************************************************************/

hti* hti_create(ht * const restrict table){
	hti *ret = malloc(sizeof(ht));
	if(ret == NULL) return NULL;
	ret->table = table;

	hti_reset(ret);

	return ret;
}





/************************************************************
hti* hti_create(hti *iter)
*************************************************************
Description: Get next value from iterator, iter->key == NULL
             if last element reached
Params in:   iter: the iterator
************************************************************/

void hti_next(hti * const restrict iter){
	do{
		NEXTENTRY(iter->current, iter->table->entrysize);
	}while(iter->current->key[0] == '\0' && iter->current <= iter->table->lastentry);
	if(iter->current > iter->table->lastentry){
		iter->current = NULL;
		iter->key = NULL;
		iter->value = NULL;
	}
	else{
		iter->key = iter->current->key;
		iter->value = iter->current->value;
	}
}





/************************************************************
hti* hti_reset(hti *iter)
*************************************************************
Description: Resets an iterator to the beginning of the table
Params in:   iter: the iterator
************************************************************/

void hti_reset(hti * const restrict iter){
	if(iter->table->length == 0){
		iter->key = NULL;
		iter->value = NULL;
		return;
	}
	for(iter->current = iter->table->entries; iter->current->key[0] == '\0'; NEXTENTRY(iter->current, iter->table->entrysize));
	iter->key = iter->current->key;
	iter->value = iter->current->value;

}







/************************************************************
*************************************************************
"PRIVATE" functions
*************************************************************
************************************************************/






/************************************************************
ht_entry* moveentries(ht *table, ht_entry *src, size_t count)
*************************************************************
Description: move "count" entries from table starting from
             src to an allocated buffer
             the elments are removed from the table
Params in:   table: the hash table
             src:   the first element to remove
             count: the number of elements to remove starting
                    from src
Returns:     the buffer containing the removed elements
             NULL on allocation failure
************************************************************/

static ht_entry* moveentries(const ht * const restrict table, ht_entry *src, const size_t count){
	ht_entry *lastentry = ENTRYINDEX(src, table->entrysize, count - 1);
	ht_entry * const restrict dst = malloc(count * table->entrysize);
	if(dst == NULL) return NULL;

	if(lastentry <= table->lastentry){
		memcpy(dst, src, count * table->entrysize);
		while(src <= lastentry){
			src->key[0] = '\0';
			NEXTENTRY(src, table->entrysize);
		}
	}
	else{
		lastentry = src + ((char*)lastentry - (char*)table->lastentry);
		memcpy(dst, src, (char*)table->lastentry - (char*)src);
		while(src <= table->lastentry) {
			src->key[0] = '\0';
			NEXTENTRY(src, table->entrysize);
		}
		src = table->entries;
		memcpy(dst + ((char*)table->lastentry - (char*)src), table->entries, (char*)lastentry - (char*)table->entries);
		while(src <= lastentry) {
			src->key[0] = '\0';
			NEXTENTRY(src, table->entrysize);
		}
	}
	return dst;
}





/************************************************************
void insertentries(ht *table, ht_entry *src, size_t count)
*************************************************************
Description: insert "count" entries from a buffer to the
             hashtable
Params in:   table: the destination hash table
             src:   an array of elements
             count: the number of elements from src array
************************************************************/

static void insertentries(const ht *const restrict table, const ht_entry * restrict src, size_t count){
	ht_entry * restrict dst;

	while(count > 0){
		size_t index = src->hash % table->capacity;
		dst = ENTRYINDEX(table->entries, table->entrysize, index);
		while(dst->key[0] != '\0'){
			NEXTENTRY(dst, table->entrysize);
			if(dst > table->lastentry) dst = table->entries;
		}
		// copying whole entry with padding will be faster (glibc will use large data instructions)
		memcpy(dst, src, table->entrysize);
		NEXTENTRY(src, table->entrysize);
		count--;
	}
}





/************************************************************
ht_entry* findentriestorelocate(ht *table, char *key, size_t *count){
*************************************************************
Description: finds the entry corresponding to key, with the
             number of consecutives elements in the array
             starting with this entry
Params in:   table: the hash table
             key:   the key to find
Params out:  count: the number of consecutives valid elements
                    in the table starting from the element
                    corresponding to the provided key
Returns:     NULL if key not found, else address of the entry
************************************************************/

static ht_entry* findentriestorelocate(const ht * const restrict table, const char * const restrict key, size_t * const restrict count){
	const uint64_t hash = hash_key(key);
	const size_t index = hash % table->capacity;
	const ht_entry *entry = ENTRYINDEX(table->entries, table->entrysize, index);
	const ht_entry *entrystart = NULL;
	
	*count = 0;
	
	while(entry->key[0] != '\0') {
		if(entrystart == NULL){
			if(strcmp(key, entry->key) == 0) entrystart = entry;
		}
		else (*count)++;
		NEXTENTRY(entry, table->entrysize);
		if(entry > table->lastentry) entry = table->entries;
	}
	//NULL if key not found
	return (ht_entry*)entrystart;
}





/************************************************************
ht_entry* alloc_entries(size_t capacity, size_t entrylength, ht_entry **lastentry)
*************************************************************
Description: allocates and initializes the array of entries
             for the hash table
Params in:   capacity:    number of elements to allocate
             entrylength: size of one element
Params out:  lastentry:   address of the last element
Returns:     NULL if alloc failed, else array of entries
************************************************************/

static ht_entry* alloc_entries(const size_t capacity, const size_t entrylength, ht_entry **lastentry){
	ht_entry * const restrict entries = malloc(capacity * entrylength);
	if(entries == NULL) return entries;
	ht_entry *entry;
	*lastentry = ENTRYINDEX(entries, entrylength, capacity - 1);
	// faster than calloc, we don't need zeroing all data
	for(entry = entries; entry <= *lastentry; NEXTENTRY(entry, entrylength))
		entry->key[0] = '\0';

	return entries;
}





/************************************************************
uint64_t hash_key(char *key)
*************************************************************
Description: computes an hash value
Params in:   key:    value to compute the hash from
Returns:     hash value
************************************************************/

static uint64_t hash_key(const char * restrict key){
	uint64_t hash = FNV_OFFSET;
	while(*key){
		hash ^= (uint64_t)*key;
		hash *= FNV_PRIME;
		key++;
	}
	return hash;
}





/************************************************************
int ht_expand(ht *table)
*************************************************************
Description: resizes an hash table (expand or shrink depending
             on fill percentage)
Params in:   table: the hash table
Returns:     1 if alloc failed, 0 on success
************************************************************/

static int ht_expand(ht* const restrict table) {
	if(table->capacity > SIZE_MAX / 2) return 1; // overflow (TODO: better tune)
	const size_t new_capacity = table->length <= table->capacity / 4 ? table->capacity / 2 : table->capacity * 2;

	ht_entry *entrysrc, *entrydst;
	const ht_entry * const restrict maxentrysrc = table->lastentry;
	
	ht_entry* const restrict new_entries = alloc_entries(new_capacity, table->entrysize, &(table->lastentry));
	if(new_entries == NULL) return 1;

	for(entrysrc = table->entries; entrysrc <= maxentrysrc; NEXTENTRY(entrysrc, table->entrysize)){
		if(entrysrc->key[0] == '\0') continue;
		const size_t index = entrysrc->hash % new_capacity;
		entrydst = ENTRYINDEX(new_entries, table->entrysize, index);
		while(entrydst->key[0] != '\0'){
			NEXTENTRY(entrydst, table->entrysize);
			if(entrydst > table->lastentry) entrydst = new_entries;
		}
		// copying whole entry with padding will be faster (glibc will use large data instructions)
		memcpy(entrydst, entrysrc, table->entrysize);
	}
	free(table->entries);
	table->entries = new_entries;
	table->capacity = new_capacity;

	return 0;
}





/************************************************************
void* setentry(ht_entry *entry, char *key, uint64_t hash, void *value, size_t valuesize)
*************************************************************
Description: fills an ht_entry structure
Params in:   entry: the structure to fill
             key:       key from (key,value) pair
             hash:      computed hash from key
             value:     value from (key,value) pair
             valuesize: like "sizeof value"
Returns:     1 if alloc failed, 0 on success
************************************************************/

static void* setentry(ht_entry * const restrict entry, const char * const restrict key, const uint64_t hash, const void * const restrict value, const size_t valuesize){
	strcpy(entry->key, key);
	memcpy(entry->value, value, valuesize);
	entry->hash = hash;
	return entry->value;
}

